<?php
/*--------------------------------------------------------------------------------------------------
    SellingUnitCachedReadService.php 2020-06-16
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2020 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Shop\SellingUnit\Unit\Services;

use Gambio\Shop\SellingUnit\Unit\SellingUnit;
use Gambio\Shop\SellingUnit\Unit\SellingUnitInterface;
use Gambio\Shop\SellingUnit\Unit\SellingUnitRepositoryInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Interfaces\QuantityInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;

/**
 * Class SellingUnitCachedReadService
 * @package Gambio\Shop\SellingUnit\Unit\Services
 */
class SellingUnitCachedReadService extends SellingUnitReadService
{
    /**
     * This value needs to be in seconds
     */
    protected const CACHE_TIME_TO_LIVE = 10;
    
    /**
     * On these pages the cached units are not loaded
     */
    protected const PROHIBITED_PAGES = [
        'request_port.php',
        'shop.php'
    ];
    
    /**
     * @var string
     */
    protected $cacheDir;
    
    /**
     * @var SellingUnitInterface[]
     */
    protected $units = [];
    
    /**
     * @var SellingUnitInterface[]
     */
    protected $excludedFromCaching = [];
    
    /**
     * @var string
     */
    protected $url;
    
    /**
     * @var string
     */
    protected $urlHash;
    
    
    /**
     * SellingUnitCachedReadService constructor.
     *
     * @param SellingUnitRepositoryInterface $repository
     */
    public function __construct(SellingUnitRepositoryInterface $repository)
    {
        parent::__construct($repository);
    
        $this->cacheDir = dirname(__DIR__, 4) . DIRECTORY_SEPARATOR . 'cache';
        $this->url      = $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        // selling unit quantities are different in same parts of the shop (e.g shopping cart)
        $this->urlHash = sha1($this->url);
    }
    
    
    /**
     * Stores initiated selling units to the cache directory
     */
    public function __destruct()
    {
        $this->removeOutdatedCacheFiles();
        
        if (count($this->units)) {
            
            foreach ($this->units as $identification => $unit) {
                
                // instances that were read from cache are excluded from caching
                if (array_key_exists($identification, $this->excludedFromCaching) === false) {
                    
                    $this->createSellingUnitCacheFile((string)$identification, $unit);
                }
            }
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function getSellingUnitBy(
        SellingUnitId $id,
        $product = null,
        $xtcPrice = null,
        QuantityInterface $quantity = null
    ): SellingUnitInterface {
    
        $result         = parent::getSellingUnitBy($id, $product, $xtcPrice, $quantity);
        $identification = (string)$result->presenter()->getPresentationIdCollection();
        
        //  SellingUnit was already initiated in this instance
        if (array_key_exists($identification, $this->units) === true) {
            
            return $this->units[$identification];
        }
        
        //  Cache file exists and is not older than defined in constant CACHE_TIME_TO_LIVE
        if ($this->validCacheFileForSellingUnitExists($identification) && false === $this->cachingIsProhibited()) {
    
            return $this->units[$identification] = $this->excludedFromCaching[$identification] = $this->readSellingUnitFromCacheFile($identification);
        }
        
        return $this->units[$identification] = $result;
    }
    
    
    /**
     * @param string $identification
     *
     * @return bool
     */
    protected function validCacheFileForSellingUnitExists(string $identification): bool
    {
        $path = $this->cacheFilePathForSellingUnit($identification);

        clearstatcache();
        
        if (is_readable($path) ) {
    
            $timestamp        = filemtime($path);
            $cacheFileIsValid = is_int($timestamp) && time() - $timestamp < self::CACHE_TIME_TO_LIVE;
            
            if ($cacheFileIsValid === false) {
                
                unlink($path);
            }
            
            return $cacheFileIsValid;
        }
        
        return false;
    }
    
    
    /**
     * @param string $identification
     *
     * @return string
     */
    protected function cacheFilePathForSellingUnit(string $identification): string
    {
        $path     = $this->cacheDir . DIRECTORY_SEPARATOR;
        $fileName = 'selling-unit-' . $this->urlHash . '-' . $identification . '.cache';
        
        return $path . $fileName;
    }
    
    
    /**
     * @param string $identification
     *
     * @return SellingUnitInterface
     */
    protected function readSellingUnitFromCacheFile(string $identification): SellingUnitInterface
    {
        $cacheContent = file_get_contents($this->cacheFilePathForSellingUnit($identification));
    
        return unserialize($cacheContent);
    }
    
    
    /**
     * @param string               $identification
     * @param SellingUnitInterface $sellingUnit
     */
    protected function createSellingUnitCacheFile(string $identification, SellingUnitInterface $sellingUnit): void
    {
        $cacheFilePath = $this->cacheFilePathForSellingUnit($identification);
        
        if (is_writable($this->cacheDir)) {
            
            $serialized    = serialize($sellingUnit);
            file_put_contents($cacheFilePath, $serialized);
            chmod($cacheFilePath, 0777);
        }
    }
    
    
    /**
     * @return bool
     */
    protected function cachingIsProhibited(): bool
    {
        $urlParts = explode('/', strtok($this->url, '?'));
        $script   = end($urlParts);
        
        return in_array($script, self::PROHIBITED_PAGES, true);
    }
    
    
    /**
     * Removes cache files that are older than the set time to live
     */
    protected function removeOutdatedCacheFiles(): void
    {
        $ttl          = self::CACHE_TIME_TO_LIVE;
        $suCacheFiles = glob($this->cacheDir . DIRECTORY_SEPARATOR . 'selling-unit*');

        clearstatcache();
        
        array_map(static function ($filePath) use ($ttl) {
    
            if (is_readable($filePath)) {
    
                $timestamp    = filemtime($filePath);
                $cacheIsValid = is_int($timestamp) && time() - $timestamp < $ttl;

                if ($cacheIsValid === false) {

                    unlink($filePath);
                }
            }
        },
            $suCacheFiles);
    }
}